--- Convert element attributes to a plain Lua table
local function attributes_to_table(element)
  local attrs = {}
  for key, value in pairs(element.attributes) do
    attrs[key] = value
  end
  return attrs
end

--- Convert a Lua value to Typst syntax
local function typst_value(value)
  if value == "true" then
    return "true"
  elseif value == "false" then
    return "false"
  elseif value == "none" then
    return "none"
  elseif tonumber(value) then
    return tostring(value)
  else
    return string.format('"%s"', value)
  end
end

--- Build attribute string for Typst function calls
local function build_attribute_string(attrs)
  local parts = {}
  for key, value in pairs(attrs) do
    table.insert(parts, string.format('%s: %s', key, typst_value(value)))
  end
  return table.concat(parts, ', ')
end

local M = {}

--- Build Typst block wrappers with optional attributes
function M.build_typst_block_wrappers(config, attrs)
  local has_attributes = next(attrs) ~= nil

  if has_attributes or config.arguments then
    local attr_string = build_attribute_string(attrs)
    return string.format('#%s(%s)[', config.wrapper, attr_string), ']'
  else
    return string.format('#%s[', config.wrapper), ']'
  end
end

--- Extract first heading as title attribute
function M.extract_first_heading_as_title(el, attrs)
  if not attrs['title'] and #el.content > 0 then
    local first_elem = el.content[1]
    if first_elem.t == 'Header' then
      attrs['title'] = pandoc.utils.stringify(first_elem.content)
      local new_content = {}
      for i = 2, #el.content do
        table.insert(new_content, el.content[i])
      end
      el.content = new_content
    end
  end
end

--- Build wrapped content for components
function M.build_wrapped_content(div, config, extract_title)
  local attrs = attributes_to_table(div)

  if extract_title then
    M.extract_first_heading_as_title(div, attrs)
  end

  local opening, closing = M.build_typst_block_wrappers(config, attrs)
  local result = { pandoc.RawBlock('typst', opening) }
  for _, item in ipairs(div.content) do
    table.insert(result, item)
  end
  table.insert(result, pandoc.RawBlock('typst', closing))

  return result
end

return M
